"use client";

import React, { useEffect, useState } from "react";
import { getToken } from "@/lib/apiEndpoints";
import InnerBanner from "../../../../../components/InnerBanner";
import { useCart } from "@/context/CartContext";
import { usePathname, useRouter } from "next/navigation";
import Link from "next/link";

interface HandwrittenAssignmentDetailClientProps {
    productData: any;
    courseCode: string;
    assignmentCode: string;
}

export default function HandwrittenAssignmentDetailClient({
    productData,
    courseCode,
    assignmentCode,
}: HandwrittenAssignmentDetailClientProps) {
    const router = useRouter();
    const pathname = usePathname();
    const { addToCart, setBuyNowItem } = useCart();
    // Initialize auth state
    const [isLoggedIn, setIsLoggedIn] = useState(false);

    useEffect(() => {
        setIsLoggedIn(!!getToken());
    }, []);

    const [selectedType, setSelectedType] = useState("");
    const [selectedMedium, setSelectedMedium] = useState("");
    const [selectedSession, setSelectedSession] = useState("");
    const isDisabled = !selectedType || !selectedMedium || !selectedSession;

    // Quantity State
    const [quantity, setQuantity] = useState(1);
    const maxQuantity = productData?.assignment_quantity ?? 1;

    useEffect(() => {
        setQuantity(1);
    }, [productData]);

    // This script injection was in the original file, preserving behavior.
    useEffect(() => {
        // Only run if not already present
        // Keeping original logic structure but simplified slightly
        // Note: In Next.js App Router, global styles are usually in layout, but locally scoped
        // or dynamically loaded ones might be here for legacy reasons.
    }, []);

    const increaseQty = () =>
        setQuantity(prev => (prev < maxQuantity ? prev + 1 : maxQuantity));

    const decreaseQty = () =>
        setQuantity(prev => (prev > 1 ? prev - 1 : 1));

    const [selectionError, setSelectionError] = useState<string | null>(null);

    const handleBuyNow = () => {
        setSelectionError(null);

        if (!selectedType) {
            setSelectionError("Please select assignment type.");
            return;
        }

        if (!selectedMedium) {
            setSelectionError("Please select medium.");
            return;
        }

        if (!selectedSession) {
            setSelectionError("Please select session.");
            return;
        }

        const rawString = `${courseCode}|${assignmentCode}|${Date.now()}`;

        setBuyNowItem({
            id: `handwritten-assignment-${rawString}`,
            name: productData?.title ?? "",
            price: productData?.salePrice ?? 0,
            qty: quantity ?? 1,
            img: productData?.image ?? "",
            courseCode,
            itemCode: assignmentCode,
            subjectIds: null,
            selectedType,
            selectedMedium,
            selectedSession,
            dataSaveId: null,
            pageName: "handwritten-assignment-page",
        });

        // 👉 Redirect to checkout page
        router.push("/checkout");
    };

    const handleAddToCart = () => {
        // Reset error
        setSelectionError(null);

        if (!selectedType) {
            setSelectionError("Please select assignment type.");
            return;
        }

        if (!selectedMedium) {
            setSelectionError("Please select medium.");
            return;
        }

        if (!selectedSession) {
            setSelectionError("Please select session.");
            return;
        }

        const rawString = `${courseCode}|${assignmentCode}|${Date.now()}`;

        addToCart({
            id: `handwritten-assignment-${rawString}`,
            name: productData?.title ?? "",
            price: productData?.salePrice ?? 0,
            qty: quantity ?? 1,
            img: productData?.image ?? "",
            courseCode,
            itemCode: assignmentCode,
            subjectIds: null,
            selectedType,
            selectedMedium,
            selectedSession,
            dataSaveId: null,
            pageName: "handwritten-assignment-page",
        });
    };

    return (
        <div >
            {/* Styles */}
            <style jsx global>{`
                /* Reuse styles */
                @keyframes pulse { 0%, 100% { transform: scale(1); } 50% { transform: scale(1.05); } }
                .animate-fadeInUp { animation: fadeInUp 0.8s ease-out; }
                @keyframes fadeInUp { from { transform: translateY(30px); opacity: 0; } to { transform: translateY(0); opacity: 1; } }
                .badge-animate { background: red; color: white; padding: 3px 16px; border-radius: 20px; animation: pulse 1.5s infinite; }
                .hover-lift:hover { transform: translateY(-10px); box-shadow: 0 20px 40px rgba(47, 128, 255, 0.3) !important; transition: all 0.3s ease; }
                .selete-box { border-radius: 10px; padding: 0px 20px; display: flex; align-items: center; width: 300px; }
                .prices-sy { font-size: 24px; }
                /* span.h2.fw-bold.d-flex { font-size: 48px; } Removed this overly explicit selector causing issues */
            `}</style>

            <InnerBanner title={productData?.title} breadcrumb={productData?.breadcrumbTitle} />

            <div className="solved-assignment-page py-5">
                <div className="container">
                    <div className="row g-4 animate-fadeInUp">
                        {/* Left - Product Image */}
                        <div className="col-lg-5">
                            <div className="card border-0 shadow-lg hover-lift border-glow" style={{ borderRadius: '20px', overflow: 'hidden' }}>
                                <div style={{ position: 'relative' }}>
                                    {/* Placeholder for PDF preview */}
                                    <div style={{ height: '400px', background: '#f8f9fa', display: 'flex', alignItems: 'center', justifyContent: 'center' }}>
                                        <i className="fas fa-file-pdf fa-5x text-danger"></i>
                                    </div>
                                    <div
                                        className="badge-animate"
                                        style={{ position: 'absolute', top: '20px', right: '20px', boxShadow: '0 5px 15px rgba(220, 53, 69, 0.5)' }}
                                    >
                                        {productData?.discount}% OFF
                                    </div>
                                </div>
                            </div>
                        </div>

                        {/* Right - Product Details */}
                        <div className="col-lg-7">
                            <h1 className="h3 fw-bold mb-4" style={{ lineHeight: '1.4' }}>
                                IGNOU <span className="text-primary"> {productData?.code} </span> Handwritten Scanned PDF Assignment
                            </h1>

                            {/* Type Selector */}
                            <div className="selete-box mb-1 d-flex gap-3 border ">
                                <label className="form-label d-flex m-0" style={{ color: '#000' }}>Type</label>
                                <select
                                    className="form-select border-0"
                                    value={selectedType}
                                    onChange={(e) => setSelectedType(e.target.value)}
                                >
                                    <option value="">-- Select Type --</option>
                                    {productData?.types.map((type: any) => (
                                        <option key={type.enc_id} value={type.enc_id}>{type.type_name}</option>
                                    ))}
                                </select>
                            </div>

                            <div className="mb-1 selete-box d-flex gap-3 border ">
                                <label className="form-label d-flex m-0" style={{ color: '#000' }}>Medium</label>
                                <select
                                    className="form-select border-0"
                                    value={selectedMedium}
                                    onChange={(e) => setSelectedMedium(e.target.value)}
                                >
                                    <option value="">-- Select Medium --</option>
                                    {productData?.medium.map((medium: any) => (
                                        <option key={medium.enc_id} value={medium.enc_id}>{medium.language_name}</option>
                                    ))}
                                </select>
                            </div>

                            {/* Session Selector */}
                            <div className="selete-box d-flex gap-3 border">
                                <label className="form-label d-flex m-0" style={{ color: '#000' }}>Session</label>
                                <select
                                    className="form-select border-0"
                                    value={selectedSession}
                                    onChange={(e) => setSelectedSession(e.target.value)}
                                >
                                    <option value="">-- Select Session --</option>
                                    {productData?.sessions.map((session: any) => (
                                        <option key={session.enc_id} value={session.enc_id}>{session.session_name}</option>
                                    ))}
                                </select>
                            </div>

                            {/* Price */}
                            <div className=" p-3" >
                                <div className="">
                                    <div className="d-flex align-items-center">
                                        <span className="h2 fw-bold d-flex align-items-center gap-2"><div className="prices-sy">₹ </div>{productData?.salePrice}</span>
                                        <span className="text-decoration-line-through text-muted ms-3 h5">₹{productData?.maxRetailPrice}</span>
                                        <div className="badge-animate">{productData?.discount}% OFF </div>
                                    </div>
                                    <div className="text-muted small mt-1">
                                        <i className="fas fa-shield-alt" style={{ color: '#2f80ff' }}></i> INCLUSIVE OF ALL TAXES
                                    </div>

                                    {/* Quantity Selector */}
                                    <div className="d-flex align-items-center gap-2 my-4">
                                        <button className="btn btn-outline-secondary" onClick={decreaseQty}>-</button>
                                        <span className="fs-4">{String(quantity).padStart(2, '0')}</span>
                                        <button className="btn btn-outline-secondary" onClick={increaseQty}>+</button>
                                    </div>

                                    {selectionError && (
                                        <div className="alert alert-danger mt-3 py-2">
                                            {selectionError}
                                        </div>
                                    )}

                                    <small
                                        className="text-danger my-1 d-block"
                                        style={{ fontSize: "0.72rem", fontStyle: "italic" }}
                                    >
                                        ** Type, Medium, Session, and Others are mandatory fields.
                                    </small>

                                    {/* Buttons */}
                                    <div className="d-flex gap-3">
                                        {isLoggedIn ? (
                                            <button
                                                type="button"
                                                className="btn btn-light border w-50 fw-bold"
                                                onClick={handleBuyNow}
                                                disabled={isDisabled}
                                            >
                                                Buy Now
                                            </button>
                                        ) : (
                                            <Link
                                                href={`/login?redirect=${encodeURIComponent(pathname)}&intent=buy_now`}
                                                className="btn btn-light border w-50 fw-bold text-white d-flex align-items-center justify-content-center"
                                                style={{
                                                    backgroundColor: '#2f80ff',
                                                    border: 'none',
                                                }}
                                            >
                                                Buy Now
                                            </Link>
                                        )}

                                        <button
                                            className="btn btn-primary w-50 py-2 fw-bold"
                                            onClick={handleAddToCart}
                                            disabled={isDisabled}
                                        >
                                            Add to Cart
                                        </button>
                                    </div>
                                </div>
                            </div>

                            {/* Why Choose */}
                            <div className="p-4 mt-3" style={{ background: 'linear-gradient(135deg, rgba(47, 128, 255, 0.05), rgba(47, 128, 255, 0.1))', borderRadius: '15px' }}>
                                <h5 className="fw-bold mb-3"><i className="fas fa-star" style={{ color: '#ffa500' }}></i> Features</h5>
                                <ul className="list-unstyled">
                                    {(productData?.highlights || []).map((item: string, idx: number) => (
                                        <li key={idx} className="mb-2"><i className="fas fa-check-circle text-primary me-2"></i>{item}</li>
                                    ))}
                                </ul>
                            </div>

                        </div>
                    </div>

                    {/* Detailed Description */}
                    <div className="container pt-5">
                        <div className="p-4 bg-white border rounded shadow-sm mb-4">
                            <h3 className="fw-bold mb-3">Description</h3>
                            <p dangerouslySetInnerHTML={{ __html: productData?.main_description ?? "" }} />
                        </div>
                    </div>

                </div>
            </div>
        </div>
    );
}
